/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.domain;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Holder for {@link ValidationErrorType}.
 */
public final class ValidationErrorTypeHolder {

    /**
     * The holder instance.
     */
    public static final ValidationErrorTypeHolder INSTANCE = new ValidationErrorTypeHolder();

    private final List<ValidationErrorType> dataList = new ArrayList<>();
    private final Map<String, ValidationErrorType> mapByCode = new HashMap<>();
    private final Map<String, ValidationErrorTypeEnum> enumByCode = new HashMap<>();

    /**
     * Initialize the data.
     *
     * @param dataList the list of data
     */
    public void setDataList(final List<ValidationErrorType> dataList) {
        this.dataList.clear();
        this.dataList.addAll(dataList);
        this.initializeCodeMap();
    }

    private void initializeCodeMap() {
        this.mapByCode.clear();
        for (final ValidationErrorType errorType : this.dataList) {
            this.mapByCode.put(errorType.getCode(), errorType);
        }
    }

    /**
     * Find a reference data by its code.
     *
     * @param code the code
     * @return the reference data
     */
    public ValidationErrorType findByCode(final String code) {
        final ValidationErrorType errorType = this.mapByCode.get(code);
        if (errorType == null) {
            throw new IllegalArgumentException("No data found with code: " + code);
        }
        return errorType;
    }

    public ValidationErrorTypeEnum findEnumByCode(String code) {
        ValidationErrorTypeEnum errorType = enumByCode.get(code);
        if (errorType == null) {
            throw new IllegalArgumentException("No data found with code: " + code);
        }
        return errorType;
    }

    private ValidationErrorTypeHolder() {
        final Map<String, String> longDescriptions = loadDescriptionsFromProperties();
        this.setDataList(Stream.of(ValidationErrorTypeEnum.values())
                .map(e -> ValidationErrorTypeHolder.createValidationErrorType(e, longDescriptions))
                .collect(Collectors.toList()));
        Stream.of(ValidationErrorTypeEnum.values()).forEach(e -> enumByCode.put(e.getCode(), e));
    }

    private static ValidationErrorType createValidationErrorType(final ValidationErrorTypeEnum errorTypeEnum, final Map<String, String> longDescriptions) {
        final ValidationErrorType validationErrorType = new ValidationErrorType();
        validationErrorType.setCode(errorTypeEnum.getCode());
        validationErrorType.setDescription(errorTypeEnum.getDescription());
        validationErrorType.setLongDescription(longDescriptions.get(errorTypeEnum.getCode()));
        validationErrorType.setValidationResultType(errorTypeEnum.getResultType());
        return validationErrorType;
    }

    private static Map<String, String> loadDescriptionsFromProperties() {
        final Properties properties = new Properties();
        try {
            properties.load(ValidationErrorTypeHolder.class.getResourceAsStream("/validation_error_descriptions.properties"));
        } catch (final IOException e) {
            throw new IllegalStateException("Failed to load validation_error_descriptions.properties file which is supposed to be in classpath", e);
        }

        final Map<String, String> result = new HashMap<>();
        final String prefix = "error.code.";
        properties.forEach((k, v) -> {
            if (!((String) k).startsWith(prefix)) {
                return;
            }
            final String code = ((String) k).substring(prefix.length());
            result.put(code, (String) v);
        });
        return result;
    }
}
